#!/usr/bin/env python3
from logging import handlers
from collections import deque, OrderedDict

import json
import time
import select
import socket
import sys
import struct
import logging
import _pyams as pyams
import fcntl
import logging
import inspect # Get function name and line number
import urllib.request, urllib.parse, urllib.error
import os
import re
import http.cookiejar
import ssl
from datetime import datetime
import subprocess
import zlib
import ipaddress

# APP ID
MONITOR_AGENT               = 6

# AMS APP STATE
AMS_APP_NEW                 = 0
AMS_APP_IN_PROCESS          = 1
AMS_APP_JOINED              = 2

# MQTT agent external message ID
MQTTD_CLIENT_CONNECT        = 1
MQTTD_CLIENT_CONNACK        = 2
MQTTD_CLIENT_SUBCRIBE       = 3
MQTTD_CLIENT_SUBACK         = 4
MQTTD_CLIENT_UNSUBCRIBE     = 5
MQTTD_CLIENT_UNSUBACK       = 6
MQTTD_CLIENT_PUBLISH        = 7
MQTTD_CLIENT_PUBACK         = 8
MQTTD_CLIENT_RECEIVE        = 9
MQTTD_BROKER_DISCON         = 10
MQTTD_BROKER_RECON          = 11
MQTTD_CLIENT_STATE_UPDATE   = 12

# Max string len of topic
MQTT_TOPIC_LEN_MAX          = 63

# Size of message header
MSG_HDR_LEN = 12

# Port of Dpcmm and Mqtt-agent
SYNC_AGENT_DPCMM_PORT       = 41352 # Need to change to other port for Telemetry SyncAgent
IPPORT_AL_DEVOPS_MQTTA      = 41601

# Timer
RETRY_CALLHOME_INTERVAL     = 30 * 1000 # 30 secs for retrying
DEFAULT_CALLHOME_INTERVAL   = 1 * 60 * 1000 # 1 min for testing
MAX_PACKET_SIZE             = 9000

LOG_FILE                    = "/flash/ovng-monitoring.log"
LOG_FILE_SIZE               = 1280000 # bytes

# Network Telemetry RestAPI v2 mapping									  
MON_AGENT_FILE              = "/flash/{0}/pkg/ovng-agent/ovng-mon-agent.cfg".format(pyams.pyams_CS_CurrRunning(0))
DELAY_CONFIG_FILE           = "/flash/{0}/pkg/ovng-agent/delay.cfg".format(pyams.pyams_CS_CurrRunning(0))
BROKER_CONFIG_FILE          = "/flash/{0}/pkg/ovng-agent/mon-broker.cfg".format(pyams.pyams_CS_CurrRunning(0))
CLIENT_CERT                 = "/flash/switch/cloud/client.crt"
FIRST_CALLHOME_TRUE         = 1
FIRST_CALLHOME_FALSE        = 0
UPDATE_REQUIRED_TRUE        = 1
UPDATE_REQUIRED_FALSE       = 0

# Report message count
REPORT_CONNECTIONDOWN_COUNT  = 100

# Status connection
CONNECTION_DOWN = False
CONNECTION_UP   = True

# Status of endpoint
ENDPOINT_UP = 1
ENDPOINT_DOWN = 2
ENDPOINT_ACTIVE = 3
ENDPOINT_NONE = 4
ENDPOINT_UP_SEND = 5

# Type of update message
DNS_TYPE = 1
NETWORK_CONTEXT_TYPE = 2

# Dhcp6 option
OPTION_CLIENTID      = 1
OPTION_ORO           = 6
OPTION_VENDOR_CLASS  = 16

cmmIP = pyams.get_cmm_ip()
interChassisIP = pyams.get_inter_chassis_ip()
current_milli_time = lambda: int(round(time.time() * 1000))
mqtt_version = "mosquitto -h | grep version | awk '{{print $3}}'"
hb_mqtt_version = subprocess.run(mqtt_version,capture_output=True,shell=True).stdout.decode('utf-8').strip()

# Debug Level
CRITICAL        = 50
FATAL           = CRITICAL
ERROR           = 40
WARNING         = 30
WARN            = WARNING
INFO            = 20
DEBUG           = 10
NOTSET          = 0

level = INFO
if sys.argv[1] == "0":
    level = INFO
    comp = False
elif sys.argv[1] == "2":
    level = INFO
    comp = True
elif sys.argv[1] == "1":
    level = DEBUG
    comp = False
elif sys.argv[1] == "3":
    level = DEBUG
    comp = True
else:
    level = DEBUG
    comp = True

# update debug and compress flag into ovng-mon-agent.cfg
with open(MON_AGENT_FILE, "r") as file:
    try:
        data = json.load(file)
    except json.JSONDecodeError as e:
        error_log = open(LOG_FILE, 'a')
        error_log.write(f'{datetime.now()} - Error loading {MON_AGENT_FILE} - {e}\n')
        error_log.close()
        quit()
    
if level == INFO:
    data["Mqtt"]["debug"] = "off"
else:
    data["Mqtt"]["debug"] = "on"
if comp:
    data["Mqtt"]["compress"] = "on"
else:
    data["Mqtt"]["compress"] = "off"

global enhance
enhance = sys.argv[2]
if enhance == "0":
    try:
        enhance = data["Mqtt"]["enhance"]
    except:
        enhance = "off"
        data["Mqtt"]["enhance"] = "off"
else:
    data["Mqtt"]["enhance"] = enhance

if enhance == "on":
    enhance = True
else:
    enhance = False

with open(MON_AGENT_FILE, "w") as file:
    json.dump(data, file, indent=4)

logger = logging.getLogger(__name__)

def initLogger(logFileName, level=level, logFileSize=LOG_FILE_SIZE):
    log_formatter = logging.Formatter(fmt='%(asctime)s %(module)s %(levelname)s: %(message)s', datefmt='%m/%d/%Y %I:%M:%S %p')
    logHandler = loggerHandler(filename=logFileName, mode='w', maxBytes=logFileSize, encoding=None, delay=0)
    logHandler.setFormatter(log_formatter)
    logger.addHandler(logHandler)
    logger.setLevel(level=level)

def SWLOG_DEBUG(message):
    func = inspect.currentframe().f_back.f_code
    logger.debug("{0}({1}): {2}".format(func.co_name, func.co_firstlineno, message))

def SWLOG_INFO(message):
    func = inspect.currentframe().f_back.f_code
    logger.info("{0}({1}): {2}".format(func.co_name, func.co_firstlineno, message))

def SWLOG_WARN(message):
    func = inspect.currentframe().f_back.f_code
    logger.warning("{0}({1}): {2}".format(func.co_name, func.co_firstlineno, message))

def SWLOG_ERROR(message):
    func = inspect.currentframe().f_back.f_code
    logger.error("{0}({1}): {2}".format(func.co_name, func.co_firstlineno, message))


class loggerHandler(handlers.RotatingFileHandler):
    def __init__(self, filename, mode='a', maxBytes=LOG_FILE_SIZE, encoding=None, delay=0):
        backupCount = 1
        handlers.RotatingFileHandler.__init__(self, filename, mode, maxBytes, backupCount, encoding, delay)

    def doRollover(self):
        if self.stream:
            self.stream.close()
            self.stream = None
        if self.backupCount > 0:
            dfn = self.rotation_filename(self.baseFilename + ".old")
            if os.path.exists(dfn):
                os.remove(dfn)
            self.rotate(self.baseFilename, dfn)
        if not self.delay:
            self.stream = self._open()


# RESTAPI handling
class RestAPI(object):
    def __init__(self, username, password, hostaddress, userAgent, encrypted_password=False):
        self.cruft      = re.compile(b'<!--.+?-->[\n]{0,1}')
        self.ws_diag    = 200
        self.retryCounter = 0
        self.rest_connected = False
        self.hostaddress = hostaddress
        self.userAgent = userAgent
        self.username = username

        if encrypted_password:
            self.password = pyams.decrypt_password(password)
        else:
            self.password = password

        self.login(self.username, self.password, self.hostaddress, self.userAgent)

    def login(self, username, password, hostaddress, userAgent):
        self.rest_connected = False
        while not self.rest_connected:
            try:
                SWLOG_INFO("Creating the REST session. username: {0}, hostaddress: {1}, userAgent: {2}".format(username, hostaddress, userAgent))
                self.connection = RestConnection(username, password, hostaddress, userAgent)
                result = self.query("auth", "", {'username':self.connection.username, 'password':self.connection.password})
                if not self.success():
                    self.rest_connected = False
                    interval = RETRY_CALLHOME_INTERVAL
                    SWLOG_WARN("Create REST session failed: {0}".format(result.get('error')))
                    SWLOG_WARN("Retry connect request after {0} seconds.".format(int(interval/1000)))
                    select.select([], [], [], interval/1000)
                    self.retryCounter += 1
                else:
                    SWLOG_INFO("Created the REST session successfully.")
                    self.rest_connected =  True
                    self.retryCounter = 0
            except:
                SWLOG_ERROR("Error when creating RestAPI session: {0}".format(sys.exc_info()))
                SWLOG_INFO("Try again")

    def query(self, domain, urn = '', args = {}):
        try:
            result = self.connection.get(domain, urn, args)
            if result is None:
                SWLOG_ERROR("Failed to get a valid response")
                return {'diag': 404, 'output': '', 'error': 'Webview Error', 'data': []}
            code = result.getcode()
            SWLOG_DEBUG("code: {0}".format(code))

            obj = self.decode_type(result.read())
        except ValueError:
            SWLOG_ERROR("Error decoding [%s]" % result.read())
            raise
        return obj["result"]

    def post(self, domain, urn = '', args = {}):
        try:
            result = self.connection.post(domain, urn, args)
            code = result.getcode()
            SWLOG_DEBUG("code: {0}".format(code))

            obj = self.decode_type(result.read())
        except ValueError:
            SWLOG_ERROR("Error decoding [%s]" % result.read())
            raise
        except:
            SWLOG_ERROR("Unexpected error: {0}".format(sys.exc_info()))
            raise
        return obj["result"]

    def decode_type(self, data):
        SWLOG_DEBUG("Raw Response: {0}".format(data))
        clean_data = self.cruft.sub('', data)
        try:
            decoded = json.loads(clean_data)
        except ValueError as error:
            SWLOG_ERROR("JSON loads failed: {}".format(error.args[0]))
            raise
        SWLOG_DEBUG("decoded: {0}".format(decoded))
        if decoded.get('result') is not None and decoded.get('result').get('diag') is not None:
            self.store_ws_diag(decoded.get('result').get('diag'))
        return decoded

    def success(self):
        return self.ws_diag == 200

    def store_ws_diag(self, ws_diag):
        if isinstance(ws_diag, str):
            self.ws_diag = int(ws_diag)
        else:
            self.ws_diag = ws_diag

    def set(self, domain, table, index_list, objects, cli, vrfName="default"):
        objCount = 0
        result = None

        if domain == "mib":
            if vrfName != "default":
                self.connection.restHeaders = {'ALU_CONTEXT': vrfName}
            data = ""
            if index_list is not None:
                for index in index_list:
                    SWLOG_DEBUG("index: {}".format(index))
                    for idxName in index.keys():
                        SWLOG_DEBUG("index name: {0} index value: {1}".format(idxName, index[idxName]))
                        data = "{0}&mibObject{1}={2}:{3}".format(data, objCount, idxName, index[idxName])
                        SWLOG_DEBUG("data: {0}".format(data))
                        objCount += 1

            for objName in objects.keys():
                value = objects[objName]
                data = "{0}&mibObject{1}={2}:{3}".format(data, objCount, objName, value)
                objCount += 1

            SWLOG_INFO("data: {0}".format(data))
            try:
                result = self.post(domain, table, data)
            except:
                SWLOG_ERROR("Post failed: {0}".format(sys.exc_info()))
                SWLOG_WARN("result: {0}".format(result))

                SWLOG_WARN("Creating another Rest session and try again")
                result = None
                self.login(self.username, self.password, self.hostaddress, self.userAgent)
                self.set(domain, table, index_list, objects, cli, vrfName)

                if not result:
                    return

            SWLOG_INFO("======================")
            SWLOG_INFO("result: {0}".format(result))

        elif domain == "cli":
            config = { "cmd" : cli}
            result = self.query('cli', 'aos', config)
            SWLOG_INFO("======================")
            SWLOG_INFO("result: {0}".format(result))

        if self.success():
            SWLOG_INFO("Rest Set successful!")
        else:
            SWLOG_ERROR("Rest Set failed: {0}".format(result.get("error")))
            if result.get("diag") == 401: # session expired
                SWLOG_INFO("Creating another Rest session")
                result["error"] = ""
                self.login(self.connection.username, self.connection.password, self.connection.hostaddress, self.connection.userAgent)
                self.set(domain, table, index_list, objects, cli, vrfName)
            if result.get("diag") == 400 and str(result.get("error")).find("System is busy") > 0:
                SWLOG_INFO("System is busy. Re-try after 15 seconds.")
                result["error"] = ""
                select.select([], [], [], 15)
                self.set(domain, table, index_list, objects, cli, vrfName)
        return result.get("error")


class RestErrorHandler(urllib.request.HTTPDefaultErrorHandler):
    def http_error_default(self, req, fp, code, msg, headers):
        try:
            result = urllib.error.HTTPError(
                req.get_full_url(), code, msg, headers, fp)
            SWLOG_ERROR("result: {}".format(result))
            SWLOG_ERROR("CODE: %d" % code)
            SWLOG_ERROR("MSG: %s" % msg)
            SWLOG_ERROR(headers)
        except:
            SWLOG_ERROR("Unexpected error: {0}".format(sys.exc_info()))

        return result


class RestConnection(object):
    def __init__(self, username, password, hostaddress, userAgent):
        self.username    = username
        self.password    = password
        self.hostaddress = hostaddress
        self.userAgent   = userAgent
        self.useport     = -1
        self.secure      = True
        self.prettylinks = True
        self.debug = True
        self.restHeaders = None

        self.cookiejar   = http.cookiejar.CookieJar()
        ctx = ssl.create_default_context()
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE
        urllib.request.install_opener(
                urllib.request.build_opener(
                        urllib.request.HTTPCookieProcessor(self.cookiejar),
                        urllib.request.HTTPSHandler(debuglevel=0, context=ctx),
                        RestErrorHandler()))

    def endpoint(self):
        return "%s://%s%s/" % (("http", "https")[self.secure == True], self.hostaddress, ('', ':' + str(self.useport))[str(self.useport) != '-1'])

    def headers(self, request):
        request.add_header('ACCEPT', 'application/vnd.alcatellucentaos+json; version=1.0')
        if self.restHeaders is not None:
            for aheader in self.restHeaders:
                request.add_header(aheader, self.restHeaders[aheader])
        self.restHeaders = None
        return request

    def get(self, domain, urn = '', args = {}):
        SWLOG_DEBUG("GET Request: [%s%s%s%s]" % (
                self.endpoint(),
                ('?domain=' + domain, domain)[self.prettylinks],
                (('', '&urn=' + urn)[urn != ''], ('/?', '/' + urn + '?')[urn != ''])[self.prettylinks],
                ('&' + urllib.parse.urlencode(args), '')[not args]))
        request = urllib.request.Request(
                "%s%s%s%s" % (
                self.endpoint(),
                ('?domain=' + domain, domain)[self.prettylinks],
                (('', '&urn=' + urn)[urn != ''], ('/?', '/' + urn + '?')[urn != ''])[self.prettylinks],
                ('&' + urllib.parse.urlencode(args), '')[not args]))
        self.cookiejar.add_cookie_header(request)
        request.add_header('User-Agent', self.userAgent)
        request = self.headers(request)
        try:
            response = urllib.request.urlopen(request, timeout=120)
            return response
        except urllib.error.URLError as e:
            SWLOG_ERROR(f"URL Error: {e.reason}")
        except urllib.error.HTTPError as e:
            SWLOG_ERROR(f"HTTP Error: {e.code} - {e.reason}")
        except socket.timeout:
            SWLOG_ERROR("Request timed out")
        except Exception as e:
            SWLOG_ERROR(f"Unexpected Error: {str(e)}")
        return None

    def post(self, domain, urn, data):
        url = "https://{0}/{1}{2}".format(self.hostaddress, domain, ('/?', '/' + urn + '?')[urn != ''])

        SWLOG_DEBUG("POST Request: [{0}]".format(url))
        SWLOG_DEBUG(data.encode("utf-8"))

        request = urllib.request.Request(
                url,
                data.encode("utf-8"),
                {'User-Agent': self.userAgent})
        self.cookiejar.add_cookie_header(request)
        request = self.headers(request)

        return urllib.request.urlopen(request, timeout=120)


class SyncAgent(object):
    def __init__(self):
        if not pyams.is_master() or not pyams.is_primary():
            quit()
        self.time_now = current_milli_time()
        self.brokerID = 0
        self.retryCounter = 0
        self.interval = 0
        self.reTryTime = 0
        self.first_connectSuccess = False
        self.mqtt_socket = None
        self.message_deque = deque( maxlen=REPORT_CONNECTIONDOWN_COUNT)
        self.connectionBroker_status = CONNECTION_DOWN
        self.brokerCfg = BROKER_CONFIG_FILE
        self.conn = None
        self.state = AMS_APP_NEW
        # RESTAPI init
        self.cfg_file_loaded = False
        self.freqTable = {}
        self.timeTable = {}
        self.vcsn = {}
        self.get_vcsn()
        self.sampleFile = ""
        self.load_sampleFile()
        self.hb_topic = self.sampleFile["Mqtt"]["heartbeat_topic"]
        self.topic = self.sampleFile["Mqtt"]["publish_topic"]
        self.threshold = 50
        self.Tdelay = 1 #Table delay
        self.Gdelay = 2 #Group delay
        self.Sdelay = 30 #Skip delay
        self.load_delay_config()
        # RESTAPI init
        cmd = "/vroot/bin/webview access enable"
        subprocess.run(cmd,capture_output=True,shell=True).stdout.decode('utf-8').strip()
        # aaa init
        cmd = "/vroot/bin/aaa authentication http local"
        subprocess.run(cmd,capture_output=True,shell=True).stdout.decode('utf-8').strip()
        self.restSession = RestAPI(self.username, self.password, "localhost", "AL On-The-Fly", self.encrypted_password)
        global skip_delay
        skip_delay = False
        global skip_convert
        skip_convert = False
        self.hb_current_time = 0
        self.macAddress = pyams.get_mac_addr()
        self.hb_content = OrderedDict()
        self.hb_report = OrderedDict()

    def get_vcsn(self):
        # Check if the client.crt file exists
        if os.path.exists(CLIENT_CERT):
            # Command to run openssl and capture the output
            cmd = [
                'openssl',
                'x509',
                '-in', CLIENT_CERT,
                '-noout',
                '-subject',
                '-nameopt', 'RFC2253'
            ]

            # Run the command and capture the output
            result = subprocess.run(cmd, capture_output=True, text=True)

            # Check if the command was successful
            if result.returncode == 0:
                subject_info = result.stdout.strip()

                # Extract SN value
                sn_line = [line for line in subject_info.split('\n') if 'SN=' in line][0]
                sn_value = sn_line.split('SN=')[1].split('/')[0].replace(",", "")
                self.vcsn = sn_value
            else:
                SWLOG_ERROR("Getting VCSN Error: {0}".format(result.stderr))
        else:
            # Command to run show chassis and capture the output
            show_chassis_cmd = [
                '/vroot/bin/show',
                'chassis'
            ]

            # Run the command and capture the output
            result = subprocess.run(show_chassis_cmd, capture_output=True, text=True)

            # Check if the command was successful
            if result.returncode == 0:
                chassis_output = result.stdout

                # Find the Serial Number in the output
                serial_number = None
                lines = chassis_output.split('\n')
                if "Master" in chassis_output:
                    for idx, line in enumerate(lines):
                        if "Master" in line and "Serial Number:" in lines[idx + 6]:
                            serial_number = lines[idx + 6].split("Serial Number:")[1].strip().replace(",", "")
                            break
                else:
                    for idx, line in enumerate(lines):
                        if "Serial Number:" in lines[idx + 5]:
                            serial_number = lines[idx + 5].split("Serial Number:")[1].strip().replace(",", "")
                            break
                if serial_number:
                    self.vcsn = serial_number
                else:
                    SWLOG_ERROR("Getting VCSN Error")
            else:
                SWLOG_ERROR("Getting VCSN Error: {0}".format(result.stderr))

    def load_delay_config(self):
        if os.path.exists(DELAY_CONFIG_FILE):
            with open(DELAY_CONFIG_FILE, 'r') as d:
                self.delayFile = json.load(d)
            self.threshold = self.delayFile["threshold"]
            self.Tdelay = self.delayFile["table_delay"]
            self.Gdelay = self.delayFile["group_delay"]
            self.Sdelay = self.delayFile["skip_delay"]
        else:
            SWLOG_INFO("File delay.cfg does not exist. Using default delay configuration")

    # MQTT handling
    def init_mqttAgent_connection(self):
        SWLOG_INFO("init_mqttAgent_connection")

        try:
            self.mqtt_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self.mqtt_socket.connect((interChassisIP, IPPORT_AL_DEVOPS_MQTTA))
        except socket.error:
            SWLOG_ERROR("Failed to connect to mqtt")
            self.mqtt_socket = None
            return False

        SWLOG_INFO("Connect to MQTT-AGENT success {0}".format(self.mqtt_socket))
        self.mqtt_socket.setblocking(0)
        self.mqtt_send_connect_request()
        return True

    def mqtt_send_connect_request(self):
        packedData = self.mqtt_form_packet(MQTTD_CLIENT_CONNECT, self.brokerCfg)
        self.mqtt_send_message(packedData)
        return True

    def mqtt_send_status_update(self):
        SWLOG_INFO("Send App State Update message to MQTT Agent. State: {0} brokerID {1}".format(self.state, self.brokerID))
        if self.brokerID == 0:
            SWLOG_WARN("Broker ID is zero, do not send update.")
            return

    def mqtt_publish(self, data, singleTopic = None, compress = False):
        if self.connectionBroker_status is CONNECTION_DOWN:
            if len(self.message_deque) < REPORT_CONNECTIONDOWN_COUNT:
                self.message_deque.append(data)
            return True

        packedData = self.mqtt_form_packet(MQTTD_CLIENT_PUBLISH, data, singleTopic, compress = compress)
        self.mqtt_send_message(packedData)
        return True

    def mqtt_subsribe(self, data):
        if self.connectionBroker_status is CONNECTION_DOWN:
            if len(self.message_deque) < REPORT_CONNECTIONDOWN_COUNT:
                self.message_deque.append(data)
            return True

        packedData = self.mqtt_form_packet(MQTTD_CLIENT_SUBCRIBE, data, recon=True)
        self.mqtt_send_message(packedData)
        return True

    def mqtt_send_message(self, data):
        if self.mqtt_socket is not None:
            try:
                self.mqtt_socket.send(data)
                #SWLOG_DEBUG("Sent {0} to {1} successful!".format(data, self.mqtt_socket.getpeername()))
                SWLOG_DEBUG("Sent to {0} successful!".format(self.mqtt_socket.getpeername()))
            except socket.error:
                #SWLOG_ERROR("Sent {0} to {1} failed!".format(data, self.mqtt_socket.getpeername()))
                SWLOG_DEBUG("Sent to {0} successful!".format(self.mqtt_socket.getpeername()))

    # Adding parameters for On-Demand model and single topic for each group
    def mqtt_form_packet(self, msgId, payload, singleTopic = None, compress = False, recon = False):
        packedData = b''
        if compress:
            SWLOG_DEBUG("Uncompressed : {0}".format(payload))
            payloadb = zlib.compress(bytes(payload, "utf-8"))
            SWLOG_DEBUG("Compressed : {0}".format(payloadb))
        else:
            payloadb = bytes(payload, "utf-8")
            SWLOG_DEBUG("Uncompressed : {0}".format(payloadb))
        payloadLen = len(payloadb)
            
        if recon:
            MQTTD_MSG_S = "I{0}sI".format(MQTT_TOPIC_LEN_MAX + 1)
            MSG_HDR_S = "IIHH{0}".format(MQTTD_MSG_S)
            msgLen =  struct.calcsize(MQTTD_MSG_S)
            return  packedData

        if payloadLen == 0:

            MQTTD_MSG_S = "I{0}sI".format(MQTT_TOPIC_LEN_MAX + 1)
            MSG_HDR_S = "IIHH{0}".format(MQTTD_MSG_S)
            msgLen =  struct.calcsize(MQTTD_MSG_S)
            if singleTopic is not None:
                packedData = struct.pack(MSG_HDR_S, msgId, msgLen, 0, MONITOR_AGENT, self.brokerID, bytes(singleTopic, "utf-8"), 0)
            else:
                packedData = struct.pack(MSG_HDR_S, msgId, msgLen, 0, MONITOR_AGENT, self.brokerID, bytes(self.topic, "utf-8"), 0)
        else :

            MQTTD_MSG_S = "I{0}sI{1}s".format(MQTT_TOPIC_LEN_MAX + 1, payloadLen)
            MSG_HDR_S = "IIHH{0}".format(MQTTD_MSG_S)
            msgLen =  struct.calcsize(MQTTD_MSG_S)
            if singleTopic is not None:
                packedData = struct.pack(MSG_HDR_S, msgId, msgLen, 0, MONITOR_AGENT, self.brokerID,  bytes(singleTopic, "utf-8") , payloadLen , payloadb)
            else:
                packedData = struct.pack(MSG_HDR_S, msgId, msgLen, 0, MONITOR_AGENT, self.brokerID,  bytes(self.topic, "utf-8") , payloadLen , payloadb)

        return  packedData

    def mqtt_msg_handler(self, buffer):
        MSG_HDR_S = "IIHH"
        (msgID, msgLen, msgFlags, msgVersion) = struct.unpack(MSG_HDR_S, buffer[:12])

        if msgID == MQTTD_CLIENT_CONNACK:
            (self.brokerID,) = struct.unpack("i", buffer[12:16])
            SWLOG_INFO("BROKER ID: {0}".format(self.brokerID))
            if self.brokerID == 0 :
                self.connectionBroker_status = CONNECTION_DOWN
                self.reTryTime = current_milli_time()
                self.interval = RETRY_CALLHOME_INTERVAL
                SWLOG_WARN("Cannot connect to the broker. Re-send connect request after {} seconds.".format(int(self.interval/1000)))
                self.retryCounter += 1
                SWLOG_DEBUG("Retry Counter:{}".format(self.retryCounter))
            else:
                self.retryCounter = 0
                if self.first_connectSuccess is False:
                    SWLOG_INFO("First connect to broker. Send report message")
                    self.first_connectSuccess = True
                    self.connectionBroker_status = CONNECTION_UP

                self.state = AMS_APP_JOINED
                self.mqtt_send_status_update()
                self.mqtt_subsribe("")
                start_time = time.time()
                self.process_timerRequest(current_milli_time(), FIRST_CALLHOME_TRUE)
                SWLOG_INFO("====Full table execute time: {0}====".format(time.time()-start_time))
                

        elif msgID == MQTTD_BROKER_DISCON:
            SWLOG_INFO("Session disconnect")
            self.connectionBroker_status = CONNECTION_DOWN

            self.state = AMS_APP_IN_PROCESS
            self.mqtt_send_status_update()

        elif msgID == MQTTD_BROKER_RECON:
            SWLOG_INFO("Session re-connect")
            self.connectionBroker_status = CONNECTION_UP
            # Send message in queue
            if len(self.message_deque) < REPORT_CONNECTIONDOWN_COUNT:
                while self.message_deque:
                    self.mqtt_publish(self.message_deque.popleft())
            # Send report message and flush queue
            else:
                self.message_deque.clear()

            self.state = AMS_APP_JOINED
            self.mqtt_send_status_update()
            self.mqtt_subsribe("")

        elif msgID == MQTTD_CLIENT_RECEIVE:
            (messagelen,) = struct.unpack("I",buffer[80:84])
            message = buffer[84:84+messagelen]
            message = message.decode()
            SWLOG_INFO("Received message from broker: len:{0}, {1}".format(messagelen,message))
            return(message)
        
        else:
            SWLOG_ERROR("Unknown message ID: {0}".format(msgID))

    def mqtt_read_packet(self, inputs, socket, data_read):
        data_read_len = len(data_read)
        if data_read_len < MSG_HDR_LEN :
            return False, data_read

        MSG_HDR_S = "IIHH"
        (msgID, msgLen, msgFlags, msgVersion) = struct.unpack(MSG_HDR_S, data_read[:12])

        data_read_len = 0

        while data_read_len < msgLen:

            readable, writable, exceptional = select.select([socket], [], [], 0.5)

            if not (readable or writable or exceptional):
                SWLOG_ERROR("Read ntCmm socket time out")
                return False, data_read

            # Handle inputs
            for s in readable:
                if s is socket:
                    data_read_temp = socket.recv(msgLen - data_read_len)

                    if data_read_temp:
                        data_read_len += len(data_read_temp)
                        data_read += data_read_temp

                    else:
                        inputs.remove(socket)
                        socket.close()
                        self.mqtt_connection_down()
                        return False, data_read
        return True, data_read
    
    def mqtt_connection_down(self):
        SWLOG_INFO("Connection with MQTT-AGENT down ")
        self.mqtt_socket = None

    def load_sampleFile(self):
        loadTable = {}
        locDict = {}
        filterDict = {}

        # Default config file
        with open(MON_AGENT_FILE, 'r') as s:
            try:
                self.sampleFile = json.load(s)
            except json.JSONDecodeError as e:
                SWLOG_ERROR(f"Error loading {MON_AGENT_FILE} - {e}\n")
                select.select([], [], [], 2)
                SWLOG_INFO(f"Try to load {MON_AGENT_FILE} one more time")
                try:
                    self.sampleFile = ""
                    self.sampleFile = json.load(s)
                except json.JSONDecodeError as e:
                    SWLOG_ERROR(f"Error loading {MON_AGENT_FILE} - {e}\n")
                    SWLOG_ERROR(f"Check {MON_AGENT_FILE} and restart the monitoring-agent")
                    quit()
            SWLOG_INFO(f"Loaded {MON_AGENT_FILE} successful")

        mqttConfig = self.sampleFile["Mqtt"]
        restCredential = mqttConfig.get("restCredential", "config-sync:43999035FF06DE228172EE59538F8AC0")
        self.username, self.password = restCredential.split(":")
        if mqttConfig.get("restEncrypted", "disable") == "enable":
            self.encrypted_password = True
        
        for key in list(self.sampleFile["Groups"]):
            locDict["frequency"] = self.sampleFile["Groups"][key]["frequency"]
            locDict["prev_time"] = current_milli_time()
            loadTable[key] = locDict.copy()

        self.timeTable = dict(sorted(loadTable.items(), key=lambda item: item[1]["frequency"]))
        SWLOG_INFO(self.timeTable)   

    def process_timerRequest(self, time_now, firstTimeFlag):
        global skip_delay
        default_topic = self.sampleFile["Mqtt"]["publish_topic"]
        
        self.heartbeat_to_mqtt(firstTimeFlag)

        for table_key, value in self.timeTable.items():
            table_list = self.sampleFile["Groups"][table_key]["Tables"]
            table_enable = self.sampleFile["Groups"][table_key]["enable"]
            table_topic = self.sampleFile["Groups"][table_key]["publish_topic"]

            if not table_topic:
                table_topic = default_topic

            if firstTimeFlag == FIRST_CALLHOME_TRUE and table_enable == UPDATE_REQUIRED_TRUE:
                self.process_restAPI_request(table_list, table_key, time_now, table_topic)
                value["prev_time"] = time_now + int(round(group_exec_time * 1000))

                SWLOG_INFO("====Timestamp: {0}====".format(value["prev_time"]))
                SWLOG_INFO("====First update new report to the MQTT server====\n")
            else:

                if time_now - value["prev_time"] >= self.timeTable[table_key]["frequency"]*DEFAULT_CALLHOME_INTERVAL and table_enable == UPDATE_REQUIRED_TRUE:
                    self.process_restAPI_request(table_list, table_key, time_now, table_topic)
                    local_prev_time = value["prev_time"]
                    value["prev_time"] = time_now + int(round(group_exec_time * 1000))

                    SWLOG_INFO("====Arrived after: {0} minutes with topic: {1}====".format((value["prev_time"]-local_prev_time)/DEFAULT_CALLHOME_INTERVAL, table_topic))
                    SWLOG_INFO("====Timestamp: {0}====".format(value["prev_time"]))
                    SWLOG_INFO("====Update one round report to the MQTT server====\n")
        skip_delay = False

    def heartbeat_to_mqtt(self, firstTimeFlag):
        time_now = time.monotonic()
        if firstTimeFlag == FIRST_CALLHOME_TRUE:
            self.hb_content = OrderedDict(deviceMac=self.macAddress, vcSerialNumber=self.vcsn, uptime=self.heartbeat_up_time(), manage_mode="OVNG")
            self.hb_report = OrderedDict(version=hb_mqtt_version, messageID=current_milli_time(), method="heartbeat", deviceMac=self.macAddress, vcSerialNumber=self.vcsn, contents=self.hb_content)
            
            self.mqtt_publish(json.dumps(dict(self.hb_report), sort_keys=False), self.hb_topic,compress=comp)
            SWLOG_INFO("====Published heartbeat succeed to topic: {0}====".format(self.hb_topic))
            self.hb_current_time = time_now
        else:
            if (time_now - self.hb_current_time)*1000 >= DEFAULT_CALLHOME_INTERVAL:
                self.hb_content["uptime"] = self.heartbeat_up_time()
                self.hb_report["messageID"] = current_milli_time()
                self.hb_report["contents"] = self.hb_content

                self.mqtt_publish(json.dumps(dict(self.hb_report), sort_keys=False), self.hb_topic,compress=comp)
                SWLOG_INFO("====Published heartbeat succeed to topic: {0}====".format(self.hb_topic))
                self.hb_current_time = time_now

    def heartbeat_up_time(self):
        hb_up_time = 0
        restUpTime = self.restSession.query("mib", "system", {"mibObject0": "sysUpTime"})

        if restUpTime["diag"] == 200 and restUpTime["data"]:
            try:
                hb_up_time = round(int(restUpTime["data"]["rows"]["sysUpTime"]) / 100)
            except KeyError as e:
                SWLOG_ERROR("Heartbeat sysUpTime value is empty")
        else:
            SWLOG_ERROR("Heartbeat sysUpTime can't be queried")

        return hb_up_time

    def cpu_usage(self):
        cmd = "/vroot/bin/show health | grep CPU | awk '{{print $2}}'"
        cpu = subprocess.run(cmd,capture_output=True,shell=True).stdout.decode('utf-8').strip()
        return int(cpu)

    def correct_ipv6_format(self, data):
        """
        Recursively traverses through the data structure to find and correct
        incorrectly formatted IPv6 addresses.
        """
        if isinstance(data, dict):
            return {k: self.correct_ipv6_format(v) for k, v in data.items()}
        elif isinstance(data, list):
            return [self.correct_ipv6_format(item) for item in data]
        elif isinstance(data, str) and data.count(':') == 15:
            # Attempt to correct the IPv6 format
            hex_str = data.replace(':', '')
            try:
                ipv6_bytes = bytes.fromhex(hex_str)
                ipv6_address = str(ipaddress.IPv6Address(ipv6_bytes))
                return ipv6_address
            except ValueError:
                # Return the original string if conversion fails
                SWLOG_INFO("Correction failed; retaining original.")
                return data
        else:
            return data

    def process_restAPI_request(self, table_list, table_key, time_stamp, singleTopic = None):
        # Dictionary for data queried by MIB domain
        global skip_delay, group_exec_time
        cpuUsageDict = {}

        start_time = time.time()
        if int(self.cpu_usage()) > self.threshold and not skip_delay:
            SWLOG_INFO("Table {0} is being delayed {1} second due to high CPU usage - {2}%".format(table_list, self.Gdelay, self.threshold))
            cpuUsageDict["deviceMac"] = self.macAddress
            cpuUsageDict["vcSerialNumber"] = self.vcsn
            cpuUsageDict["cpuUsageAlert"] = "Table {0} is being delayed {1} second due to high CPU usage - {2}%".format(table_list, self.Gdelay, self.threshold)
            self.mqtt_publish(json.dumps(cpuUsageDict, sort_keys=False), self.topic,compress=comp)

        while int(self.cpu_usage()) > self.threshold and not skip_delay:
            select.select([], [], [], self.Gdelay)
            if time.time() - start_time > self.Sdelay:
                SWLOG_INFO("CPU continuously at high usage over {0} seconds - Skip delay".format(self.Sdelay))
                cpuUsageDict["deviceMac"] = self.macAddress
                cpuUsageDict["vcSerialNumber"] = self.vcsn
                cpuUsageDict["cpuUsageAlert"] = "CPU continuously at high usage over {0} seconds - Skip delay".format(self.Sdelay)
                self.mqtt_publish(json.dumps(cpuUsageDict, sort_keys=False), self.topic,compress=comp)
                skip_delay = True
                break
        
        mibResDict = {}
        if enhance:
            TempmibResDict = {}
        start_time = time.time()
        for table in table_list:
            self.heartbeat_to_mqtt(0)
            SWLOG_INFO("Table pres: {0}".format(table))
            try:
                tempTable = self.sampleFile["Tables"][table]
            except KeyError as e:
                SWLOG_ERROR(f"Agent stopped reason: Tables define error {e}")
                SWLOG_ERROR(f"Check table {e} in ovng-mon-agent.cfg and restart the monitoring-agent")
                quit()
            if len(tempTable) > 1:
                if enhance:
                    TempmibResDict = self.processFileInputTable(tempTable["table"], tempTable["objects"].split(", "))
                    try:
                        mibResDict[table+"_cols"] = TempmibResDict["cols"]
                        mibResDict[table+"_rows"] = TempmibResDict["rows"]
                    except:
                        try:
                            mibResDict[table] = TempmibResDict["rows"]
                        except:
                            mibResDict[table] = TempmibResDict
                else:
                    mibResDict[table] = self.processFileInputTable(tempTable["table"], tempTable["objects"].split(", "))
            else:
                self.processFileInputCmd(tempTable["cmd"], table_key, time_stamp, start_time, singleTopic)
        temp = mibResDict
        mibResDict = self.correct_ipv6_format(mibResDict)
        if temp != mibResDict:
            SWLOG_INFO("Correction ipv6 format succeed.")
        if len(mibResDict) > 0:
            self.rest_publish_to_mqtt(table_key, time_stamp, mibResDict, start_time, singleTopic)
        group_exec_time = time.time() - start_time
        SWLOG_INFO("Execution time: {0}".format(group_exec_time))

    def rest_publish_to_mqtt(self, table_key, time_stamp, publishedDict, exec_time = 0, singleTopic = None):
        # Dictionary for storing data queried within a group concept
        groupResDict =  {}
        end_time = time.time() - exec_time
        
        groupResDict["type"] = table_key
        groupResDict["deviceMac"] = self.macAddress
        groupResDict["vcSerialNumber"] = self.vcsn
        groupResDict["timeStamp"] = time_stamp + int(round(end_time * 1000))
        groupResDict["execTime"] = end_time
        groupResDict["data"] = publishedDict

        self.mqtt_publish(json.dumps(groupResDict, sort_keys=False), singleTopic, compress=comp)
        SWLOG_INFO("====Published succeed table {0}!!====".format(table_key))

    def convert_format(self, resTbl):
        # Check if "data" is empty
        if resTbl["data"] == []:
            return resTbl
        # Check if resTbl is a list
        if isinstance(resTbl, list):
            return resTbl
        # Check if "data" is present and contains "rows"
        if not resTbl.get("data", {}).get("rows"):
            return resTbl
            
        rows = resTbl["data"]["rows"]
        # If "rows" is a list, skip conversion
        if isinstance(rows, list):
            return resTbl
            
        # Extract the first row to get the column names (keys)
        first_row = next(iter(resTbl["data"]["rows"].values()))
        if not isinstance(first_row, dict):
            # If "rows" contains flat key-value pairs (non-dictionaries), skip conversion
            return resTbl
        cols = list(first_row.keys())

        # Transform rows from dicts to lists, preserving the order
        rows = OrderedDict()
        for key, row in resTbl["data"]["rows"].items():
            # Convert each dictionary into a list based on the column order
            rows[key] = [row[col] for col in cols]

        # Create a new "data" section with "cols" first, "rows" second, and other entries last
        data_section = OrderedDict()
        data_section["cols"] = cols
        data_section["rows"] = rows

        # Now, add any other keys from the original "data" section (excluding "cols" and "rows")
        for key, value in resTbl["data"].items():
            if key not in ["rows"]:
                data_section[key] = value

        # Create the resTbl_new structure, preserving the key order
        resTbl_new = OrderedDict([
            ("domain", resTbl["domain"]),
            ("diag", resTbl["diag"]),
            ("output", resTbl["output"]),
            ("error", resTbl["error"]),
            ("data", data_section)  # Use the new ordered "data" section
        ])
        return resTbl_new  # Return the converted JSON

    def processFileInputTable(self, table, objectList):
        objCounter = 0
        objReqList = {}
        errorDict = {}
        processDict = {}
        processList = []
        global skip_convert
        temp = 1

        for obj in objectList:
            mibObjTxt = "mibObject" + str(objCounter)
            objReqList[mibObjTxt] = obj
            objCounter = objCounter + 1

        nextIndex = True
        while nextIndex != False:
            objReqList["limit"] = 50
            objReqList["ignoreError"] = "true"
            resTbl = self.restSession.query("mib", table, objReqList)
            if resTbl["error"] == "You must login first":
                SWLOG_INFO("REST API Session timeout")
                SWLOG_INFO("Create new Session")
                self.restSession = RestAPI(self.username, self.password, "localhost", "AL On-The-Fly", self.encrypted_password)
                SWLOG_INFO("Retry query table one more time")
                resTbl = self.restSession.query("mib", table, objReqList)
            if resTbl["error"] == "no such session - expired?":
                SWLOG_INFO("REST API Session timeout")
                SWLOG_INFO("Create new Session")
                self.restSession = RestAPI(self.username, self.password, "localhost", "AL On-The-Fly", self.encrypted_password)
                SWLOG_INFO("Retry query table one more time")
                resTbl = self.restSession.query("mib", table, objReqList)
            #print("wait {0}".format(self.Tdelay))
            select.select([], [], [], self.Tdelay)
            if enhance:
                skip_convert = False
                oldresTbl = resTbl
                resTbl = self.convert_format(resTbl)
                temp = 2
                if resTbl == oldresTbl:
                    skip_convert = True

            if resTbl["diag"] == 200 and resTbl["data"]:
                try:
                    if type(resTbl["data"]["rows"]) != list:
                        if enhance and not skip_convert:
                            if processDict == {}:
                                processDict["cols"] = resTbl["data"]["cols"]
                                processDict["rows"] = resTbl["data"]["rows"]
                            else:
                                processDict["rows"].update(resTbl["data"]["rows"])
                        else:
                            processDict.update(resTbl["data"]["rows"])
                    else:
                        processList.extend(resTbl["data"]["rows"])
                except KeyError as e:
                    SWLOG_ERROR("Switch query table {0} error".format(table))
                    SWLOG_ERROR("Switch Query memory usage over threshold")

                if len(resTbl["data"]) > temp:
                    objReqList["startIndex"] = resTbl["data"]["nextIndex"]
                else:
                    nextIndex = False
            else:
                if len(resTbl["error"]) > 0:
                    errorDict["query_error"] = resTbl["error"]
                else:
                    errorDict["query_error"] = "MIB {0} is either empty or unsupported on this platform".format(table)
                processDict.update(errorDict)
                nextIndex = False
        
        if len(processDict) > 0:
            return processDict
        else:
            return processList

    # Process RestAPI CLI domain with limit option as multiple queries
    def processFileInputCmd(self, cmd, table_key, time_stamp, exec_time = 0, singleTopic = None):
        errorDict = {}
        ov_flag = False
        ov_args = {}
        
        while True:
            retCmd = self.mappingCmdLimit(cmd, ov_flag, ov_args)
            newResCmd = "/vroot/bin/" + retCmd
            resCmd = subprocess.run(newResCmd.split(" "), capture_output=True).stdout.decode("utf-8", "ignore")
            
            try:
                json.loads(resCmd)
            except json.decoder.JSONDecodeError as e:
                SWLOG_ERROR("Error parsing JSON data: {0}".format(e))
                SWLOG_ERROR("Check the data around line {0}, column {1}".format(e.lineno, e.colno))
                return

            if "ERROR" not in resCmd and json.loads(resCmd)["next"]["has_next"] != 0:  
                ov_args.clear()
                js_resCmd = json.loads(resCmd)
                self.rest_publish_to_mqtt(table_key, time_stamp, js_resCmd, exec_time, singleTopic)
                
                ov_flag = True
                ov_args = js_resCmd["next"]["next_index"]
                continue
            else:
                if "ERROR" not in resCmd and json.loads(resCmd)["next"]["has_next"] == 0:
                    js_resCmd = json.loads(resCmd)
                    self.rest_publish_to_mqtt(table_key, time_stamp, js_resCmd, exec_time, singleTopic)
                else:
                    try:
                        errorDict["query_error"] = resCmd["error"]
                    except TypeError as e:
                        # Handle the error gracefully
                        SWLOG_ERROR("An error occurred: {0}".format(e))
                        errorDict["query_error"] = resCmd
                    self.rest_publish_to_mqtt(table_key, time_stamp, errorDict, exec_time, singleTopic)
                break
            
    def mappingCmdLimit(self, cmd, ov_flag, ov_args):
        retCmd = cmd + " limit 2000"
        ext_args = None
        
        if ov_flag == True and len(ov_args) > 0:
            if cmd == "debug show json-mac-learning":
                ext_args = " domain {0} locale {1} interface {2} service {3} subid {4} mac {5}".format(ov_args["c1"], ov_args["c2"], ov_args["c3"], ov_args["c4"], ov_args["c5"], ov_args["c6"])
            elif cmd == "debug show json-isis-spb-unicast":
                ext_args = " bvlan {0} topology {1} mac {2}".format(ov_args["c1"], ov_args["c2"], ov_args["c3"])
            elif cmd == "debug show json-isis-spb-spf":
                ext_args = " bvlan {0} sysmac {1}".format(ov_args["c1"], ov_args["c2"])
            elif cmd == "debug show svcmgr json-sdp-info":
                ext_args = " sdp {0}".format(ov_args["c1"])
            elif cmd == "debug show svcmgr json-sdp-bind":
                ext_args = " svcid {0} sdpid {1}".format(ov_args["c1"], ov_args["c2"])
            elif cmd == "debug show json-ip-interface":
                ext_args = " ifindex {0}".format(ov_args["c1"])
            elif cmd == "debug show json-lldp-remote":
                ext_args = " port-number {0} time-mark {1} index {2}".format(ov_args["c1"], ov_args["c2"], ov_args["c3"])
            elif cmd == "debug show json-alcether-stats":
                ext_args = " interface {0}".format(ov_args["c1"])
            elif cmd == "debug show json-interface-stats":
                ext_args = " index {0}".format(ov_args["c1"])
            elif cmd == "debug show interfaces json-dot3-stats":
                ext_args = " ifindex {0}".format(ov_args["c1"])
            elif cmd == "debug show spb isis json-adjacency-static":
                ext_args = " topology {0} ifindex {1}".format(ov_args["c1"], ov_args["c2"])
            elif cmd == "debug show spb isis json-adjacency-dynamic":
                ext_args = " topology {0} ifindex {1} peer-sys-id {2}".format(ov_args["c1"], ov_args["c2"], ov_args["c3"])
            elif cmd == "debug show svcmgr json-port-profile":
                ext_args = " port-profile-id {0}".format(ov_args["c1"])
            elif cmd == "debug show svcmgr json-port":
                ext_args = " port-id {0}".format(ov_args["c1"])
            elif cmd == "debug show svcmgr json-sap":
                ext_args = " service {0} port {1} encap {2}".format(ov_args["c1"], ov_args["c2"], ov_args["c3"])
            elif cmd == "debug show svcmgr json-svc":
                ext_args = " service {0}".format(ov_args["c1"])
            else:
                ext_args = " vlan {0}".format(ov_args["c1"])
            
            retCmd = retCmd + ext_args
        
        return retCmd

def main():
    try:
        lockfp = open('/tmp/ntagent.pid', 'w')
        fcntl.lockf(lockfp, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        # Another instance is running
        SWLOG_ERROR("NT Agent is already running!")
        sys.exit(0)
    
    syncAgent = SyncAgent()

    # Contain all socket
    inputs = []

    # Connect to MQTT-AGENT
    if syncAgent.init_mqttAgent_connection() == True:
        inputs.append(syncAgent.mqtt_socket)

    # Loop for wait message
    while True:

        readable, writable, exceptional = select.select(inputs, [], inputs, 0.05)

        # Handle polling interval
        if not (readable or writable or exceptional):
            # get current time
            syncAgent.time_now = current_milli_time()
            syncAgent.process_timerRequest(syncAgent.time_now, FIRST_CALLHOME_FALSE)

            # Call when connect to broker failed
            if syncAgent.reTryTime > 0 and syncAgent.time_now - syncAgent.reTryTime >= syncAgent.interval:
                SWLOG_INFO("Send connection request")
                syncAgent.reTryTime = 0
                syncAgent.mqtt_send_connect_request()

        # Handle read socket
        for s in readable:

            #For MQTT-AGENT
            if s is syncAgent.mqtt_socket:
                data_read = s.recv(MSG_HDR_LEN)

                if data_read:
                    retcode, data_read = syncAgent.mqtt_read_packet(inputs, s, data_read)
                    if retcode == True:
                        syncAgent.mqtt_msg_handler(data_read)
                else:
                    inputs.remove(s)
                    s.close()
                    syncAgent.mqtt_connection_down()

        for s in exceptional:
            if s is syncAgent.mqtt_socket:
                inputs.remove(s)
                s.close()
                syncAgent.mqtt_connection_down()

        # Re-connect to mqtt-agent
        if syncAgent.mqtt_socket is None:
            SWLOG_INFO("Re-connect to MQTT-AGENT")
            if syncAgent.init_mqttAgent_connection() == True:
                inputs.append(syncAgent.mqtt_socket)

if __name__ == '__main__':
    initLogger(LOG_FILE)
    main()
