#!/usr/bin/env python3
from logging import handlers
from collections import deque, OrderedDict

import json
import time
import select
import socket
import sys
import struct
import logging
import _pyams as pyams
import fcntl
import logging
import inspect # Get function name and line number


import os
import re
import subprocess

# APP ID
CONFIG_AGENT                = 7

# AMS APP STATE
AMS_APP_NEW                 = 0
AMS_APP_IN_PROCESS          = 1
AMS_APP_JOINED              = 2

# MQTT agent external message ID
MQTTD_CLIENT_CONNECT        = 1
MQTTD_CLIENT_CONNACK        = 2
MQTTD_CLIENT_SUBCRIBE       = 3
MQTTD_CLIENT_SUBACK         = 4
MQTTD_CLIENT_UNSUBCRIBE     = 5
MQTTD_CLIENT_UNSUBACK       = 6
MQTTD_CLIENT_PUBLISH        = 7
MQTTD_CLIENT_PUBACK         = 8
MQTTD_CLIENT_RECEIVE        = 9
MQTTD_BROKER_DISCON         = 10
MQTTD_BROKER_RECON          = 11
MQTTD_CLIENT_STATE_UPDATE   = 12

# Max string len of topic
MQTT_TOPIC_LEN_MAX          = 63

# Size of message header
MSG_HDR_LEN = 12

# PortMqtt-agent
IPPORT_AL_DEVOPS_MQTTA      = 41601

# Timer
RETRY_CALLHOME_INTERVAL     = 30 * 1000 # 30 secs for retrying

CFG_FILE                    = "/flash/{0}/pkg/ovng-agent/ovng-con-agent.cfg".format(pyams.pyams_CS_CurrRunning(0))
LOG_FILE                    = "/flash/ovng-config.log"
LOG_FILE_SIZE               = 1280000 # bytes
BROKER_CONFIG_FILE          = "/flash/{0}/pkg/ovng-agent/con-broker.cfg".format(pyams.pyams_CS_CurrRunning(0))
CLIENT_CERT                 = "/flash/switch/cloud/client.crt"

# Report message count
REPORT_CONNECTIONDOWN_COUNT  = 100

# Status connection
CONNECTION_DOWN = False
CONNECTION_UP   = True

cmmIP = pyams.get_cmm_ip()
interChassisIP = pyams.get_inter_chassis_ip()
current_milli_time = lambda: int(round(time.time() * 1000))
passwords = []

# Debug Level
CRITICAL        = 50
FATAL           = CRITICAL
ERROR           = 40
WARNING         = 30
WARN            = WARNING
INFO            = 20
DEBUG           = 10
NOTSET          = 0

#LOG_LEVEL	= DEBUG
LOG_LEVEL	= INFO

logger = logging.getLogger(__name__)
# update debug and compress flag into ovng-mon-agent.cfg
with open(CFG_FILE, "r") as file:
    data = json.load(file)
if sys.argv[1] == "1":
    level = DEBUG
    data["debug"] = "on"
else:
    level = INFO
    data["debug"] = "off"
with open(CFG_FILE, "w") as file:
    json.dump(data, file, indent=4)

def initLogger(logFileName, level=level, logFileSize=LOG_FILE_SIZE):
    log_formatter = logging.Formatter(fmt='%(asctime)s %(module)s %(levelname)s: %(message)s', datefmt='%m/%d/%Y %I:%M:%S %p')
    logHandler = loggerHandler(filename=logFileName, mode='a', maxBytes=logFileSize, encoding=None, delay=0)
    logHandler.setFormatter(log_formatter)
    logger.addHandler(logHandler)
    logger.setLevel(level=level)

def SWLOG_DEBUG(message):
    func = inspect.currentframe().f_back.f_code
    logger.debug("{0}({1}): {2}".format(func.co_name, func.co_firstlineno, message))

def SWLOG_INFO(message):
    func = inspect.currentframe().f_back.f_code
    logger.info(message)

def SWLOG_WARN(message):
    func = inspect.currentframe().f_back.f_code
    logger.warning("{0}({1}): {2}".format(func.co_name, func.co_firstlineno, message))

def SWLOG_ERROR(message):
    func = inspect.currentframe().f_back.f_code
    logger.error("{0}({1}): {2}".format(func.co_name, func.co_firstlineno, message))

def filterPassword(command):
    global passwords
    newCommand = command  # Start with the original command

    for keyword in ["password", "key"]:
        if keyword in newCommand:
            # Check for the keyword enclosed in escaped double or single quotes
            match = re.findall(rf'{keyword}\s+\\["\']([^\\["\']+?)\\["\']', newCommand)
            if match:
                passwords.extend(match)  # Add found passwords to the list
                newCommand = re.sub(rf'{keyword}\s+\\["\']([^\\["\']+?)\\["\']', f'{keyword} ******', newCommand)
            
            # Check for the keyword enclosed in single or double quotes
            match = re.findall(rf'{keyword}\s+[\'"]([^\'"]+)[\'"]', newCommand)
            if match:
                passwords.extend(match)  # Add found passwords to the list
                newCommand = re.sub(rf'{keyword}\s+[\'"]([^\'"]+)[\'"]', f'{keyword} ******', newCommand)
            
            # Check for the keyword without any quotes
            match = re.findall(rf'{keyword}\s+([^\'"\s]+)', newCommand)
            if match:
                passwords.extend(match)  # Add found passwords to the list
                newCommand = re.sub(rf'{keyword}\s+([^\'"\s]+)', f'{keyword} ******', newCommand)

    return newCommand

def hidePassword(log):
    global passwords
    for word in passwords:
        if word in log:
            log = log.replace(word, '*****')
    return log

class loggerHandler(handlers.RotatingFileHandler):
    def __init__(self, filename, mode='a', maxBytes=LOG_FILE_SIZE, encoding=None, delay=0):
        backupCount = 1
        handlers.RotatingFileHandler.__init__(self, filename, mode, maxBytes, backupCount, encoding, delay)

    def doRollover(self):
        if self.stream:
            self.stream.close()
            self.stream = None
        if self.backupCount > 0:
            dfn = self.rotation_filename(self.baseFilename + ".old")
            if os.path.exists(dfn):
                os.remove(dfn)
            self.rotate(self.baseFilename, dfn)
        if not self.delay:
            self.stream = self._open()



class SyncAgent(object):
    def __init__(self):
        if not pyams.is_master() or not pyams.is_primary():
            quit()
        self.time_now = self.hb_start_time = current_milli_time()
        self.brokerID = 0
        self.retryCounter = 0
        self.interval = 0
        self.reTryTime = 0
        self.first_connectSuccess = False
        self.mqtt_socket = None
        self.message_deque = deque( maxlen=REPORT_CONNECTIONDOWN_COUNT)
        self.connectionBroker_status = CONNECTION_DOWN
        self.vcsn = {}
        self.brokerInfo = {}
        self.get_vcsn()
        self.load_configFile()
        self.conn = None
        self.state = AMS_APP_NEW
        self.macAddress = pyams.get_mac_addr()
        self.output = False

    def get_vcsn(self):
        # Check if the client.crt file exists
        if os.path.exists(CLIENT_CERT):
            # Command to run openssl and capture the output
            cmd = [
                'openssl',
                'x509',
                '-in', CLIENT_CERT,
                '-noout',
                '-subject',
                '-nameopt', 'RFC2253'
            ]

            # Run the command and capture the output
            result = subprocess.run(cmd, capture_output=True, text=True)

            # Check if the command was successful
            if result.returncode == 0:
                subject_info = result.stdout.strip()

                # Extract SN value
                sn_line = [line for line in subject_info.split('\n') if 'SN=' in line][0]
                sn_value = sn_line.split('SN=')[1].split('/')[0].replace(",", "")
                self.vcsn = sn_value
            else:
                SWLOG_ERROR("Getting VCSN Error: {0}".format(result.stderr))
        else:
            # Command to run show chassis and capture the output
            show_chassis_cmd = [
                '/vroot/bin/show',
                'chassis'
            ]

            # Run the command and capture the output
            result = subprocess.run(show_chassis_cmd, capture_output=True, text=True)

            # Check if the command was successful
            if result.returncode == 0:
                chassis_output = result.stdout

                # Find the Serial Number in the output
                serial_number = None
                lines = chassis_output.split('\n')
                if "Master" in chassis_output:
                    for idx, line in enumerate(lines):
                        if "Master" in line and "Serial Number:" in lines[idx + 6]:
                            serial_number = lines[idx + 6].split("Serial Number:")[1].strip().replace(",", "")
                            break
                else:
                    for idx, line in enumerate(lines):
                        if "Serial Number:" in lines[idx + 5]:
                            serial_number = lines[idx + 5].split("Serial Number:")[1].strip().replace(",", "")
                            break
                if serial_number:
                    self.vcsn = serial_number
                else:
                    SWLOG_ERROR("Getting VCSN Error")
            else:
                SWLOG_ERROR("Getting VCSN Error: {0}".format(result.stderr))


    def load_configFile(self):
        with open(CFG_FILE, 'r') as f:
            self.config_file = json.load(f)

        self.config_version = 1

        if self.config_version == 1:
            self.topic = self.config_file["subscribe_topics"]+"/"+self.vcsn
            self.publishtopic = self.config_file["publish_topics"]

            self.brokerCfg = BROKER_CONFIG_FILE
            with open(BROKER_CONFIG_FILE, 'r') as file:
                for line in file:
                    if line.startswith("-h") or line.startswith("-p"):
                        keyInfo, valueInfo = line.strip().split(maxsplit=1)
                        self.brokerInfo[keyInfo] = valueInfo 

    # MQTT handling
    def init_mqttAgent_connection(self):
        SWLOG_INFO("init_mqttAgent_connection")

        try:
            self.mqtt_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self.mqtt_socket.connect((interChassisIP, IPPORT_AL_DEVOPS_MQTTA))
        except socket.error:
            SWLOG_ERROR("Failed to connect to mqtt")
            self.mqtt_socket = None
            return False

        SWLOG_INFO("Connect to MQTT-AGENT success {0}".format(self.mqtt_socket))
        self.mqtt_socket.setblocking(0)
        self.mqtt_send_connect_request()
        return True

    def mqtt_send_connect_request(self):
        packedData = self.mqtt_form_packet(MQTTD_CLIENT_CONNECT, self.brokerCfg)
        self.mqtt_send_message(packedData)
        return True

    def mqtt_send_status_update(self):
        SWLOG_INFO("Send App State Update message to MQTT Agent. State: {0} brokerID {1}".format(self.state, self.brokerID))
        if self.brokerID == 0:
            SWLOG_WARN("Broker ID is zero, do not send update.")
            return

        data = str(self.state)
        packedData = self.mqtt_form_packet(MQTTD_CLIENT_STATE_UPDATE, data)
        self.mqtt_send_message(packedData)

    def mqtt_publish(self, data, topic=None):
        if not self.mqtt_check_broker_connection():
            SWLOG_WARN("Connection to Broker is down, message is added to queue.")
            if len(self.message_deque) < REPORT_CONNECTIONDOWN_COUNT:
                self.message_deque.append(data)
            return True

        packedData = self.mqtt_form_packet(MQTTD_CLIENT_PUBLISH, data, topic)
        self.mqtt_send_message(packedData)
        return True

    def mqtt_subsribe(self, data):
        if self.connectionBroker_status is CONNECTION_DOWN:
            if len(self.message_deque) < REPORT_CONNECTIONDOWN_COUNT:
                self.message_deque.append(data)
            return True

        packedData = self.mqtt_form_packet(MQTTD_CLIENT_SUBCRIBE, data)
        self.mqtt_send_message(packedData)
        return True

    def mqtt_send_message(self, data):
        if self.mqtt_socket is not None:
            try:
                self.mqtt_socket.send(data)
                SWLOG_DEBUG("Sent {0} to {1} successful!".format(data, self.mqtt_socket.getpeername()))
            except socket.error:
                SWLOG_ERROR("Sent {0} to {1} failed!".format(data, self.mqtt_socket.getpeername()))

    def mqtt_form_packet(self, msgId, payload, topic=None):
        if topic == None:
            topic=self.topic

        packedData = b''
        payloadLen = len(payload)

        if payloadLen == 0:

            MQTTD_MSG_S = "I{0}sI".format(MQTT_TOPIC_LEN_MAX + 1)
            MSG_HDR_S = "IIHH{0}".format(MQTTD_MSG_S)
            msgLen =  struct.calcsize(MQTTD_MSG_S)
            packedData = struct.pack(MSG_HDR_S, msgId, msgLen, 0, CONFIG_AGENT, self.brokerID, bytes(topic, "utf-8"), 0)
        else :

            MQTTD_MSG_S = "I{0}sI{1}s".format(MQTT_TOPIC_LEN_MAX + 1, payloadLen)
            MSG_HDR_S = "IIHH{0}".format(MQTTD_MSG_S)
            msgLen =  struct.calcsize(MQTTD_MSG_S)
            packedData = struct.pack(MSG_HDR_S, msgId, msgLen, 0, CONFIG_AGENT, self.brokerID,  bytes(topic, "utf-8") , payloadLen , bytes(payload, "utf-8"))

        return  packedData

    def mqtt_check_broker_connection(self, retry=1):
        tried = 0
        cmd = 'echo exit | telnet {0} {1}'.format(self.brokerInfo["-h"], self.brokerInfo["-p"])
        
        while tried < retry:
            try:
                resCmd = subprocess.run(cmd, shell=True, timeout=5, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)

                if resCmd.returncode == 0:
                    SWLOG_INFO("Broker connection to {0} was reached".format(self.brokerInfo["-h"]))
                    return True
                    
                SWLOG_INFO("Broker connection to {0} is unreachable".format(self.brokerInfo["-h"]))
            except subprocess.TimeoutExpired:
                SWLOG_INFO("Broker connection to {0} is unreachable".format(self.brokerInfo["-h"]))
            except Exception as e:
                SWLOG_ERROR("Unexpected error: {0}".format(e))

            tried += 1
            time.sleep(1)

        return False

    def mqtt_msg_handler(self, buffer):
        #print("Received message from MQTT Agent: {0}".format(buffer))
        MSG_HDR_S = "IIHH"
        (msgID, msgLen, msgFlags, msgVersion) = struct.unpack(MSG_HDR_S, buffer[:12])

        if msgID == MQTTD_CLIENT_CONNACK:
            (self.brokerID,) = struct.unpack("i", buffer[12:16])
            SWLOG_INFO("BROKER ID: {0}".format(self.brokerID))
            if self.brokerID == 0 :
                # Check Broker connection
                while not self.mqtt_check_broker_connection():
                    SWLOG_WARN("Broker connection is down.")
                    time.sleep(1)

                self.connectionBroker_status = CONNECTION_DOWN
                self.reTryTime = current_milli_time()
                self.interval = RETRY_CALLHOME_INTERVAL
                SWLOG_WARN("Cannot connect to the broker. Re-send connect request after {} seconds.".format(int(self.interval/1000)))
                self.retryCounter += 1
                SWLOG_DEBUG("Retry Counter:{}".format(self.retryCounter))
            else:
                self.retryCounter = 0
                if self.first_connectSuccess is False:
                    SWLOG_INFO("First connect to broker. Send report message")
                    self.first_connectSuccess = True
                    self.connectionBroker_status = CONNECTION_UP

                self.state = AMS_APP_JOINED
                self.mqtt_send_status_update()
                self.mqtt_subsribe("")

        elif msgID == MQTTD_BROKER_DISCON:
            SWLOG_INFO("Session disconnect")
            self.connectionBroker_status = CONNECTION_DOWN

            self.state = AMS_APP_IN_PROCESS
            self.mqtt_send_status_update()

        elif msgID == MQTTD_BROKER_RECON:
            SWLOG_INFO("Session re-connect")
            self.connectionBroker_status = CONNECTION_UP
            # Send message in queue
            if len(self.message_deque) < REPORT_CONNECTIONDOWN_COUNT:
                while self.message_deque:
                    self.mqtt_publish(self.message_deque.popleft(), self.publishtopic)
            # Send report message and flush queue
            else:
                self.message_deque.clear()

            self.state = AMS_APP_JOINED
            self.mqtt_send_status_update()
            self.mqtt_subsribe("")

        elif msgID == MQTTD_CLIENT_RECEIVE:
            (messagelen,) = struct.unpack("I",buffer[80:84])
            message = buffer[84:84+messagelen]
            message = message.decode()
            SWLOG_INFO("Received message from broker: len:{0}, {1}".format(messagelen, filterPassword(message)))
            return(message)

        else:
            SWLOG_ERROR("Unknown message ID: {0}".format(msgID))

        return("")

    def mqtt_read_packet(self, inputs, socket, data_read):
        data_read_len = len(data_read)
        if data_read_len < MSG_HDR_LEN :
            return False, data_read

        MSG_HDR_S = "IIHH"
        (msgID, msgLen, msgFlags, msgVersion) = struct.unpack(MSG_HDR_S, data_read[:12])

        data_read_len = 0

        while data_read_len < msgLen:

            readable, writable, exceptional = select.select([socket], [], [], 0.5)

            if not (readable or writable or exceptional):
                SWLOG_ERROR("Read ntCmm socket time out")
                return False, data_read

            # Handle inputs
            for s in readable:
                if s is socket:
                    data_read_temp = socket.recv(msgLen - data_read_len)

                    if data_read_temp:
                        data_read_len += len(data_read_temp)
                        data_read += data_read_temp

                    else:
                        inputs.remove(socket)
                        socket.close()
                        self.mqtt_connection_down()
                        return False, data_read
        return True, data_read

    def mqtt_connection_down(self):
        SWLOG_INFO("Connection with MQTT-AGENT down ")
        self.mqtt_socket = None

    def errorRequest(self,error):
        start_time = time.time()
        ResDict={}
        ResDict["deviceMac"] = pyams.get_mac_addr()
        ResDict["vcSerialNumber"] = self.vcsn
        ResDict["Status"]="Error"
        ResDict["Error"]=error
        end_time = time.time()
        ResDict["timeStamp"] = int(round(end_time * 1000))
        ResDict["execTime"] = end_time - start_time
        return(ResDict)

    def runRequest(self,request):
        start_time = time.time()
        topic=self.publishtopic
        SWLOG_INFO("got request from OVNG {0}".format(filterPassword(request)))
        try:
            request=json.loads(request)
        except:
            error="Bad Request, request not in json format"
            ResDict=self.errorRequest(error)
            self.mqtt_publish(json.dumps(ResDict, sort_keys=False, indent=5),topic)
            SWLOG_ERROR("Bad Request, request not in json format {0}".format(filterPassword(request)))
            return
        try:
            tid=request['TransactionID']
            actions=request['actions']
            if "resp_topic" in request.keys():
                  topic=request["resp_topic"]
        except:
            error="Bad Request, request does not have TransactionID and actions"
            ResDict=self.errorRequest(error)
            self.mqtt_publish(json.dumps(ResDict, sort_keys=False, indent=5),topic)
            SWLOG_ERROR("Bad Request, request does not have TransactionID and actions")
            return
        try:
            self.output=request['showOutput']
            if self.output == "true" or self.output == "True":
                self.output = True
        except:
            self.output = False

        ResDict={}
        ResDict["deviceMac"] = pyams.get_mac_addr()
        ResDict["vcSerialNumber"] = self.vcsn
        ResDict["TransactionID"]=tid
        ResDict["Response"] = {}

        count=1
        error_count=0
        for action in actions:
            SWLOG_INFO("Running action {0}".format(filterPassword(str(action))))
            (result,err) = self.runAction(action)
            if err:
                 error_count += 1
            entry = "action" + str(count)
            ResDict["Response"][entry]={}
            actionstring=str(action)
            ResDict["Response"][entry]["action"]=actionstring
            ResDict["Response"][entry]["Status"]=result["Status"]
            ResDict["Response"][entry]["Error"]=str(result["Error"])
            if self.output:
                try:
                    ResDict["Response"][entry]["Output"]=str(result["Output"])
                except:
                    ResDict["Response"][entry]["Output"]=""

            count += 1

        end_time = time.time()
        ResDict["timeStamp"] = int(round(end_time * 1000))
        ResDict["execTime"] = end_time - start_time
        if error_count == 0:
            ResDict["Status"]="Success"
            ResDict["Error"]=""
        else:
            if error_count == (count - 1):
                 ResDict["Status"]="Error"
                 ResDict["Error"]="All actions failed"
            else:
                 ResDict["Status"]="Error"
                 ResDict["Error"]="{0} action(s) failed out of {1}".format(error_count,count-1)
 
        self.mqtt_publish(json.dumps(ResDict, sort_keys=False, indent=5),topic)
        SWLOG_INFO("====Published succeed transaction {0}====\n====Status: {status}!!====".format(ResDict["TransactionID"],
                    status="All actions success" if error_count==0 else ResDict["Error"]))

    def parse_command(self, command):
        # Regular expression to match arguments
        regex = r'(".*?"|\S+)'
        
        # Find all matches
        matches = re.findall(regex, command)
        
        parsed_args = []
        for match in matches:
            if match.startswith('"') and match.endswith('"'):
                content = match[1:-1]
                if ' ' in content:
                    # Remove quotes if the string contains spaces
                    parsed_args.append(content)
                else:
                    # Keep the quotes if the string does not contain spaces
                    parsed_args.append(match)
            else:
                parsed_args.append(match)
        
        return parsed_args

    def runAction(self, action):
        try:
            cmd=action["cmd"]
        except:
            result={"Status":"Error","Error":"Bad action, does not have a command"}
            SWLOG_ERROR("Bad action, does not have a command {0}".format(action))
            return(result,1)

        input_data = action.get("input", "")
        command = "/vroot/bin/" + cmd
        global passwords

        try:
            p = subprocess.run(command, capture_output=True, shell=True, input=input_data, text=True)
            returncode = p.returncode
            err = p.stderr
            out = p.stdout
            errors = ["Invalid entry", "Command syntax error", "Incomplete command"]

            if any(error in out for error in errors):
                SWLOG_INFO("Command run failed: {0}".format(hidePassword(out)))
                SWLOG_INFO("Retry one more time")
                # Regular expression to split command while keeping quoted parts together
                command_list = self.parse_command(command)
                p = subprocess.run(command_list, capture_output=True, input=input_data, text=True)
                returncode = p.returncode
                err = p.stderr
                out = p.stdout

            confirmation_patterns = re.compile(r'(Y/N|yes/no|y/n|Yes/No)', re.IGNORECASE)
            if confirmation_patterns.search(out) and not input_data:
                error_message = "The command requires a confirmation (Y/N)"
                SWLOG_ERROR("The command \"{0}\" requires a confirmation (yes/no)".format(filterPassword(cmd)))
                return {"Status": "Error", "Error": error_message}, 1

            if returncode > 0:
                return {"Status": "Error", "Error": "Unknown command"}, 1
            if self.output:
                result = {"Error": "", "Status": "Success", "Output": out}
            else:
                result = {"Error": "", "Status": "Success"}

            error_found = False
            SWLOG_INFO("result: {0}".format(hidePassword(out)))
            passwords = []
            m = re.search("ERROR: (.+)", out)
            if m:
                result["Error"] = m.group(1)
                result["Status"] = "Error"
                if self.output:
                    result["Output"] = ""
                error_found = True
            return result, int(error_found)

        except subprocess.CalledProcessError as e:
            error_message = "Subprocess failure: {0}".format(e)
            SWLOG_ERROR("The command \"{0}\" failed".format(filterPassword(cmd)))
            if self.output:
                return {"Status": "Error", "Error": error_message, "Output": ""}, 1
            else:
                return {"Status": "Error", "Error": error_message}, 1

def main():
    try:
        lockfp = open('/tmp/ovconfigagent.pid', 'w')
        fcntl.lockf(lockfp, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        # Another instance is running
        SWLOG_ERROR("OVNG Config Agent is already running!")
        sys.exit(0)
    
    syncAgent = SyncAgent()

    # Contain all socket
    inputs = []

    # Connect to MQTT-AGENT
    if syncAgent.init_mqttAgent_connection() == True:
        inputs.append(syncAgent.mqtt_socket)

    # Loop for wait message
    while True:

        readable, writable, exceptional = select.select(inputs, [], inputs, 0.05)

        # Handle polling interval
        if not (readable or writable or exceptional):
            # get current time
            syncAgent.time_now = current_milli_time()
            # Call when connect to broker failed
            if syncAgent.reTryTime > 0 and syncAgent.time_now - syncAgent.reTryTime >= syncAgent.interval:
                SWLOG_INFO("Send connection request")
                syncAgent.reTryTime = 0
                syncAgent.mqtt_send_connect_request()

        # Handle read socket
        for s in readable:

            #For MQTT-AGENT
            if s is syncAgent.mqtt_socket:
                data_read = s.recv(MSG_HDR_LEN)
                
                if data_read:
                    retcode, data_read = syncAgent.mqtt_read_packet(inputs, s, data_read)
                    if retcode == True:
                        request=syncAgent.mqtt_msg_handler(data_read)
                        if request !=  "":
                            syncAgent.runRequest(request)
                else:
                    inputs.remove(s)
                    s.close()
                    syncAgent.mqtt_connection_down()

        for s in exceptional:
            if s is syncAgent.mqtt_socket:
                inputs.remove(s)
                s.close()
                syncAgent.mqtt_connection_down()

        # Re-connect to mqtt-agent
        if syncAgent.mqtt_socket is None:
            SWLOG_INFO("Re-connect to MQTT-AGENT")
            if syncAgent.init_mqttAgent_connection() == True:
                inputs.append(syncAgent.mqtt_socket)

if __name__ == '__main__':
    initLogger(LOG_FILE)
    main()
    
